/* Copyright (C) 2014-2018 RealVNC Ltd.  All Rights Reserved.
 */

#ifndef UUID_21ba1e4e_02b6_11e4_abac_1f54ddde3bbf
#define UUID_21ba1e4e_02b6_11e4_abac_1f54ddde3bbf

/**
 * \file vncwifidisplaysinkcallbacks.h
 *
 * This file defines types for pointers to callback functions that may be
 * provided to VNC WiFi Display Sink instances created by the VNC WiFi Display
 * SDK. You should normally include vncwifidisplaysdk.h instead of including
 * this file directly.
 *
 * All of the callbacks in the VNCWiFiDisplaySinkCallbacks structure are
 * optional. However, you will almost certainly want to provide a
 * VNCWiFiDisplaySinkErrorCallback(), because otherwise it will be difficult to
 * terminate your application safely.
 *
 * \section section_callbacks_threading Threading
 *
 * Almost all VNC WiFi Display Sink callbacks are always invoked by the WiFi
 * Display sink thread. You must therefore avoid including code in your
 * callbacks that may disrupt this thread's activities. See \ref section_thread
 * for more information.
 *
 * The only exception to this rule is the VNCWiFiDisplaySinkLogCallback(),
 * which the SDK may make from any thread. The SDK may also call the
 * VNCWiFiDisplaySinkLogCallback() before the WiFi Display sink thread is
 * started. However, the SDK guarantees that no two threads will enter a
 * VNCWiFiDisplaySinkLogCallback() at the same time, even if there are multiple
 * VNC WiFi Display Sink instances.
 *
 * \see VNCWiFiDisplaySinkCallbacks, VNCWiFiDisplaySinkLogCallback
 */

#include "vncwifidisplaytypes.h"

#ifdef __cplusplus
extern "C"
{
#endif

/**
 * \brief Called by the SDK to allow logging from the SDK's internals.
 *
 * The information logged by this callback is intended for use by application
 * developers and RealVNC support engineers. It is not localized.
 *
 * Use this callback in conjunction with VNCWiFiDisplayParameterLog to select
 * and gather logging information from the SDK.
 *
 * \note The SDK may invoke this callback from any thread, including calls made
 * before the WiFi Display sink thread is started. However, the SDK guarantees
 * that no two threads will enter a VNCWiFiDisplaySinkLogCallback() at the same
 * time, even if there are multiple VNC WiFi Display Sink instances.
 *
 * \note It is common for VNCWiFiDisplaySinkLogCallback implementations to
 * require a context (e.g. to retrieve a file handle to which to write the
 * message). If this is the case in your application, then you should ensure
 * that you call VNCWiFiDisplaySinkSetContext() immediately after
 * VNCWiFiDisplaySinkCreate() and before calling any other SDK API.
 *
 * \param sink The VNC WiFi Display Sink instance.
 * \param context The context pointer associated with the VNC WiFi Display Sink
 * instance by VNCWiFiDisplaySinkSetContext().
 * \param category The category of the log message, as a null-terminated UTF-8
 * string.
 * \param severity The severity of the log message (see
 * VNCWiFiDisplayParameterLog for an explanation).
 * \param text The text of the log message, as a null-terminated UTF-8 string.
 *
 * \see VNCWiFiDisplaySinkCreate, VNCWiFiDisplayParameterLog
 */
typedef void VNCCALL
VNCWiFiDisplaySinkLogCallback(VNCWiFiDisplaySink *sink,
                              void *context,
                              const char *category,
                              vnc_int32_t severity,
                              const char *text);

/**
 * \brief Called by the SDK to notify of an error.
 *
 * If you explicitly stop a running VNC WiFi Display Sink with
 * VNCWiFiDisplaySinkStop(), then the VNCWiFiDisplaySinkErrorCallback() will be
 * invoked with the error code VNCWiFiDisplayErrorStopped. This allows you to
 * use the same cleanup code for this case as for error cases.
 *
 * The WiFi Display sink thread always makes exactly one call to this callback
 * just before it exits. You can therefore also use this callback alongside
 * VNCWiFiDisplaySinkThreadStartedCallback() to monitor the lifetime of the
 * WiFi Display sink thread.
 * 
 * \param sink The VNC WiFi Display Sink instance.
 * \param context The context pointer associated with the VNC WiFi Display Sink
 * instance by VNCWiFiDisplaySinkSetContext().
 * \param error The error code.
 *
 * \see VNCWiFiDisplayError, VNCWiFiDisplaySinkCreate
 * \see VNCWiFiDisplaySinkStop, VNCWiFiDisplaySinkThreadStartedCallback
 */
typedef void VNCCALL
VNCWiFiDisplaySinkErrorCallback(VNCWiFiDisplaySink *sink,
                                void *context,
                                VNCWiFiDisplayError error);

/**
 * \brief Called by the SDK immediately after the WiFi Display sink thread has
 * started.
 *
 * This callback is guaranteed to be the first callback from the WiFi Display
 * sink thread and is is primarily intended for use when implementing langauge
 * bindings (such as JNI) which require native threads to be registered.
 *
 * \param sink The VNC WiFi Display Sink instance.
 * \param context The context pointer associated with the VNC WiFi Display Sink
 * instance by VNCWiFiDisplaySinkSetContext().
 *
 * \see VNCWiFiDisplaySinkCreate, VNCWiFiDisplaySinkErrorCallback
 */
typedef void VNCCALL
VNCWiFiDisplaySinkThreadStartedCallback(VNCWiFiDisplaySink *sink,
                                        void *context);

/**
 * \brief Called by the SDK immediately after a WiFi Display decoder is
 * instantiated.
 *
 * Once this callback has been invoked, you may set decoder properties (such as
 * overlay position and size) using VNCWiFiDisplaySinkSetDecoderProperty().
 *
 * \param sink The VNC WiFi Display Sink instance.
 * \param context The context pointer associated with the VNC WiFi Display Sink
 * instance by VNCWiFiDisplaySinkSetContext().
 *
 * \see VNCWiFiDisplaySinkSetDecoderProperty
 */
typedef void VNCCALL
VNCWiFiDisplaySinkDecoderCreatedCallback(VNCWiFiDisplaySink *sink,
                                         void *context);

/**
 * \brief Called by the SDK when the WiFi Display source has selected, or
 * re-selected, the UIBC MirrorLink configuration for the WiFi Display session.
 *
 * If the WiFi Display source is a MirrorLink device, then it may select the
 * MirrorLink configuration to use for the UIBC session. The WiFi Display
 * source may set or re-select the UIBC MirrorLink configuration at any time.
 * When this occurs, the SDK invokes this callback to notify your application.
 *
 * The VNCWiFiDisplayUIBCMirrorLinkConfiguration instance provided in this
 * callback shall describe the MirrorLink configuration that shall be used for
 * the UIBC session. The selected configuration values shall be compatible with
 * those associated with this VNC WiFi Display Sink with
 * VNCWiFiDisplaySinkSetUIBCMirrorLinkConfiguration().
 *
 * \param sink The VNC WiFi Display Sink instance.
 * \param context The context pointer associated with the VNC WiFi Display Sink
 * instance by VNCWiFiDisplaySinkSetContext().
 * \param configuration The selected UIBC MirrorLink configuration.
 * \param configurationSize The size of configuration.
 * \param uibcPort The TCP port that the WiFi Display source shall listen on
 * for UIBC connections.
 *
 * \see VNCWiFiDisplaySinkSetUIBCMirrorLinkConfiguration
 * \see VNCWiFiDisplaySinkUIBCStateChangedCallback
 */
typedef void VNCCALL
VNCWiFiDisplaySinkUIBCMirrorLinkConfigurationSelectedCallback(
                VNCWiFiDisplaySink *sink,
                void *context,
                const VNCWiFiDisplayUIBCMirrorLinkConfiguration *configuration,
                size_t configurationSize,
                vnc_uint16_t uibcPort);

/**
 * \brief Called by the SDK when the state of the VNC WiFi Display Sink has
 * changed.
 *
 * The VNC WiFi Display Sink's state indicates whether the WiFi Display session
 * is being established, and whether or not the RTP stream from the WiFi
 * Display source is active.
 *
 * When the WiFi Display Sink thread is not running, the VNC WiFi Display Sink
 * assumes a state of VNCWiFiDisplaySinkStateNone. When the WiFi Display Sink
 * thread starts, the state shall change to VNCWiFiDisplayStateInit. During
 * this state, the WiFi Display Sink thread opens a TCP connection to the WiFi
 * Display Source's RTSP control port and establishes the RTSP and WiFi Display
 * sessions. Once the WiFi Display session has been established, the state
 * shall transition to VNCWiFiDisplayStreamStatePlaying. At this point, the RTP
 * media stream from the WiFi Display source shall be active.
 *
 * During the WiFi Display session, the RTP media stream may be paused at any
 * time (either at the request of the WiFi Display source, or because
 * VNCWiFiDisplaySinkPause() was invoked). When this occurs, the VNC WiFi
 * Display Sink's state shall transition to VNCWiFiDisplayStatePaused. If and
 * when the RTP stream is resumed, the state shall transition back to
 * VNCWiFiDisplayStatePlaying.
 *
 * \param sink The VNC WiFi Display Sink instance.
 * \param context The context pointer associated with the VNC WiFi Display Sink
 * instance by VNCWiFiDisplaySinkSetContext().
 * \param state The new state of the VNC WiFi Display Sink.
 */
typedef void VNCCALL
VNCWiFiDisplaySinkStateChangedCallback(VNCWiFiDisplaySink *sink,
                                       void *context,
                                       VNCWiFiDisplayState state);

/**
 * \brief Called by the SDK when the UIBC state has changed.
 *
 * The UIBC state indicates whether or not a UIBC connection is currently active.
 * When active, your application may call APIs and receive callbacks which utilise
 * the UIBC connection if a valid UIBC configuration has been selected.
 *
 * If the VNC WiFi Display Sink does not support UIBC sessions, then the UIBC state
 * shall remain at the default of VNCWiFiDisplaySinkUIBCStateNone. Otherwise, the
 * UIBC state shall change to VNCWiFiDisplaySinkUIBCStateInactive when the
 * VNC WiFi Display Sink is started, and to VNCWiFiDisplaySinkUIBCStateActive when
 * the UIBC connection is established. The SDK shall invoke this callback to notify
 * your application of any state transitions that occur.
 *
 * \param sink The VNC WiFi Display Sink instance.
 * \param context The context pointer associated with the VNC WiFi Display Sink
 * instance by VNCWiFiDisplaySinkSetContext().
 * \param state The new UIBC state.
 *
 * \see VNCWiFiDisplaySinkSetUIBCMirrorLinkConfiguration
 * \see VNCWiFiDisplaySinkUIBCMirrorLinkConfigurationSelectedCallback
 */
typedef void VNCCALL
VNCWiFiDisplaySinkUIBCStateChangedCallback(
                                  VNCWiFiDisplaySink *sink,
                                  void *context,
                                  VNCWiFiDisplaySinkUIBCState state);

/** 
 * \brief Called by the SDK when a MirrorLink source status event is received
 * from a UIBC connection.
 *
 * The SDK notifies your application when a source status event is received
 * from the MirrorLink WiFi Display source with this callback. You may send
 * sink status events to the source with
 * VNCWiFiDisplaySinkSendMirrorLinkSinkStatusEvent().
 *
 * \param sink The VNC WiFi Display Sink instance.
 * \param context The context pointer associated with the VNC WiFi Display Sink
 * instance by VNCWiFiDisplaySinkSetContext().
 * \param deviceStatus A bit-mask representing the received device status. Use
 * the VNCDeviceStatusFeature enumeration to interpret the bit-mask values.
 * \param deviceStatusSize The size of deviceStatus.
 *
 * \see VNCWiFiDisplaySinkUIBCStateChangedCallback
 * \see VNCWiFiDisplaySinkSendMirrorLinkSinkStatusEvent
 */
typedef void VNCCALL
VNCWiFiDisplaySinkMirrorLinkSourceStatusEventCallback(
                                           VNCWiFiDisplaySink *sink,
                                           void *context,
                                           const VNCDeviceStatus *deviceStatus,
                                           size_t deviceStatusSize);

/**
 * \brief Called by the SDK when a MirrorLink UI context event is received from
 * a UIBC connection.
 *
 * The SDK notifies your application when a UI context event is received from
 * the MirrorLink WiFi Display source with this callback. You may send UI
 * blocking events associated with this context event with
 * VNCWiFiDisplaySinkSendMirrorLinkUIBlockingEvent(). You must use the context
 * reference provided in this callback to associate a UI blocking event with
 * this context event.
 *
 * \param sink The VNC WiFi Display Sink instance.
 * \param context The context pointer associated with the VNC WiFi Display Sink
 * instance by VNCWiFiDisplaySinkSetContext().
 * \param contextReference A reference value for this UI context event. You may
 * use this value to send associated UI blocking events.
 * \param info An array of UI context information entries for this event.
 * \param infoSize The size of a single entry in info.
 * \param infoCount The number of entries in info.
 *
 * \see VNCWiFiDisplaySinkUIBCStateChangedCallback
 * \see VNCWiFiDisplaySinkSendMirrorLinkUIBlockingEvent
 */
typedef void VNCCALL
VNCWiFiDisplaySinkMirrorLinkUIContextEventCallback(
                      VNCWiFiDisplaySink *sink,
                      void *context,
                      vnc_uint32_t contextReference,
                      const VNCWiFiDisplayMirrorLinkUIContextInformation *info,
                      size_t infoSize,
                      size_t infoCount);

/**
 * \brief Called by the SDK when a MirrorLink audio context event is received
 * from a UIBC connection.
 *
 * The SDK notifies your application when an audio context event is received
 * from the MirrorLink WiFi Display source with this callback. You may send
 * audio blocking events associated with this context event with
 * VNCWiFiDisplaySinkSendMirrorLinkAudioBlockingEvent(). You must use the
 * context reference provided in this callback to associate an audio blocking
 * event with this context event.
 *
 * \param sink The VNC WiFi Display Sink instance.
 * \param context The context pointer associated with the VNC WiFi Display Sink
 * instance by VNCWiFiDisplaySinkSetContext().
 * \param contextReference A reference value for this audio context event. You
 * may use this value to send associated audio blocking events.
 * \param info An array of audio context information entries for this event.
 * \param infoSize The size of a single entry in info.
 * \param infoCount The number of entries in info.
 *
 * \see VNCWiFiDisplaySinkUIBCStateChangedCallback
 * \see VNCWiFiDisplaySinkSendMirrorLinkAudioBlockingEvent
 */
typedef void VNCCALL
VNCWiFiDisplaySinkMirrorLinkAudioContextEventCallback(
                   VNCWiFiDisplaySink *sink,
                   void *context,
                   vnc_uint32_t contextReference,
                   const VNCWiFiDisplayMirrorLinkAudioContextInformation *info,
                   size_t infoSize,
                   size_t infoCount);

/**
 * \brief Called by the SDK when a Mirrorlink source cut text event is received
 * from a UIBC connection.
 *
 * The SDK notifies your application when a source cut text event is received
 * from the MirrorLink WiFi Display source with this callback. This notifies
 * your application that the MirrorLink WiFi Display source host's clipboard
 * contents have been updated. If you have indicated support for UIBC cut text
 * events in the configuration set in
 * VNCWiFiDisplaySinkSetUIBCMirrorLinkConfiguration(), then you must handle
 * this event appropriately.
 *
 * You can use this notification to update the local clipboard contents so that
 * they match the host's. This allows the user to copy text from the host's
 * clipboard to a local application. You may also send sink cut text events
 * with VNCWiFiDisplaySinkSendMirrorLinkSinkCutTextEvent().
 *
 * \param sink The VNC WiFi Display Sink instance.
 * \param context The context pointer associated with the VNC WiFi Display Sink
 * instance by VNCWiFiDisplaySinkSetContext().
 * \param clipboardText The text that has been copied to the host's clipboard.
 * This text is a null-terminated UTF-8 string. It is owned by the SDK, which
 * may overwrite or free it after this callback returns.
 *
 * \see VNCWiFiDisplaySinkUIBCStateChangedCallback
 * \see VNCWiFiDisplaySinkSendMirrorLinkSinkCutTextEvent
 */
typedef void VNCCALL
VNCWiFiDisplaySinkMirrorLinkSourceCutTextEventCallback(
                                                    VNCWiFiDisplaySink *sink,
                                                    void *context,
                                                    const char *clipboardText);

/**
 * \brief Called by the SDK when a MirrorLink text output event is received
 * from a UIBC connection.
 *
 * The SDK notifies your application when a text output event is received from
 * the MirrorLink WiFi Display source with this callback. The textual
 * information remains valid for the identified application until it is either
 * overridden with a new message or invalidated. Multiple valid textual
 * information entries can exist in parallel for different uniquely
 * identifiable applications.
 *
 * The text output event is deprecated in MirrorLink 1.3. SDK users should
 * not expect this callback in MirrorLink 1.3 or later sessions.
 *
 * \param sink The VNC WiFi Display Sink instance.
 * \param context The context pointer associated with the VNC WiFi Display Sink
 * instance by VNCWiFiDisplaySinkSetContext().
 * \param applicationUniqueId The unique identifier for the application
 * emitting the text output.
 * \param text The text output provided as a null-terminated UTF-8 string, or
 * NULL if the previous text output should be cleared.
 */
typedef void VNCCALL
VNCWiFiDisplaySinkMirrorLinkTextOutputEventCallback(
                                              VNCWiFiDisplaySink *sink,
                                              void *context,
                                              vnc_uint32_t applicationUniqueId,
                                              const char *text);

/**
 * \brief Called by the SDK when the WiFi Display Decoder calls the
 * VNCWiFiDisplayDecoderNotifyResolutionSelected supporting API.
 *
 * This information can be used to correctly scale UIBC touch events to
 * session coordinates.
 *
 * \param sink The VNC WiFi Display Sink instance.
 * \param context The context pointer associated with the VNC WiFi Display Sink
 * instance by VNCWiFiDisplaySinkSetContext().
 * \param width The selected X resolution in pixels.
 * \param height The selected Y resolution in pixels.
 */
typedef void VNCCALL
VNCWiFiDisplaySinkResolutionSelectedCallback(VNCWiFiDisplaySink *sink,
                                             void *context,
                                             vnc_uint16_t width,
                                             vnc_uint16_t height);

/**
 * \brief Structure that holds pointers to callbacks invoked by the SDK.
 *
 * Any of the pointers in this structure may be NULL if you choose not to
 * implement a particular callback.  However, you will almost certainly want to
 * provide a VNCWiFiDisplaySinkErrorCallback(), because otherwise it will be
 * difficult to terminate your application safely.
 *
 * For a description of each of the callbacks, please see the documentation for
 * the corresponding typedef.
 *
 * \see VNCWiFiDisplaySinkCreate
 */
typedef struct
{
  /**
   * \brief Called by the SDK to allow logging from the SDK's internals.
   */
  VNCWiFiDisplaySinkLogCallback *logCallback;

  /**
   * \brief Called by the SDK to notify of an error.
   */
  VNCWiFiDisplaySinkErrorCallback *errorCallback;

  /**
   * \brief Called by the SDK immediately after the WiFi Display sink thread
   * has started.
   */
  VNCWiFiDisplaySinkThreadStartedCallback *threadStartedCallback;

  /**
   * \brief Called by the SDK immediately after a WiFi Display decoder plugin
   * has been created.
   */
  VNCWiFiDisplaySinkDecoderCreatedCallback *decoderCreatedCallback;

  /**
   * \brief Called by the SDK when the WiFi Display source has selected, or
   * re-selected, the UIBC MirrorLink configuration for the WiFi Display
   * session.
   */
  VNCWiFiDisplaySinkUIBCMirrorLinkConfigurationSelectedCallback *uibcMirrorLinkConfigurationSelectedCallback;

  /**
   * \brief Called by the SDK when the state of the VNC WiFi Display Sink has
   * changed.
   */
  VNCWiFiDisplaySinkStateChangedCallback *stateChangedCallback;

  /**
   * \brief Called by the SDK when the UIBC state has changed.
   */
  VNCWiFiDisplaySinkUIBCStateChangedCallback *uibcStateChangedCallback;

  /**
   * \brief Called by the SDK when a MirrorLink source status event is received
   * from a UIBC connection.
   */
  VNCWiFiDisplaySinkMirrorLinkSourceStatusEventCallback *mirrorLinkSourceStatusEventCallback;

  /**
   * \brief Called by the SDK when a MirrorLink UI context event is received
   * from a UIBC connection.
   */
  VNCWiFiDisplaySinkMirrorLinkUIContextEventCallback *mirrorLinkUIContextEventCallback;

  /**
   * \brief Called by the SDK when a MirrorLink audio context event is received
   * from a UIBC connection.
   */
  VNCWiFiDisplaySinkMirrorLinkAudioContextEventCallback *mirrorLinkAudioContextEventCallback;

  /**
   * \brief Called by the SDK when a Mirrorlink source cut text event is
   * received from a UIBC connection.
   */
  VNCWiFiDisplaySinkMirrorLinkSourceCutTextEventCallback *mirrorLinkSourceCutTextEventCallback;

  /**
   * \brief Called by the SDK when a MirrorLink text output event is received
   * from a UIBC connection.
   */
  VNCWiFiDisplaySinkMirrorLinkTextOutputEventCallback *mirrorLinkTextOutputEventCallback;

  /**
   * \brief Called by the SDK when an H.264 codec has been selected, and the
   * session resolution is known.
   */
  VNCWiFiDisplaySinkResolutionSelectedCallback *resolutionSelectedCallback;

} VNCWiFiDisplaySinkCallbacks;

#ifdef __cplusplus
}
#endif

#endif  // UUID_21ba1e4e_02b6_11e4_abac_1f54ddde3bbf
